export declare enum AD_REQUEST_METHOD {
    XHR = "xhr",
    JSONP = "jsonp" /** @deprecated use xhr instead */
}

export declare enum AD_TYPE {
    LINEAR = "linear",
    NON_LINEAR = "nonlinear"
}

export declare enum AD_VARIANT {
    CLOSER = "closer",
    DIVIDER = "divider",
    OPENER = "opener",
    COMMERCIAL = "commercial",
    SPONSORED = "sponsored"
}

export declare interface AdBreakSetup {
    adBreakPosition?: number;
    [AD_VARIANT.CLOSER]?: SeparationClips;
    [AD_VARIANT.DIVIDER]?: SeparationClips;
    [AD_VARIANT.OPENER]?: SeparationClips;
    originConfig?: AdBreakSetup | undefined;
    repeat?: number | null;
    response?: string;
    responseXML?: Document | null;
    requested?: boolean;
    tag?: string | string[];
    type?: AD_TYPE.LINEAR | AD_TYPE.NON_LINEAR;
    urls?: string[];
}

/**
 @type PublicAdAPI

 @property {string|null|Array<string|null>} adId - An ad server-defined identifier string for the ad. Array in
 wrapper chain case.
 @property {Array<string|null>} adIds - adId(s) of the ad creative node(s) in the creatives node..
 @property {string|null} adServingId - A unique or pseudo-unique (long enough to be unique when combined with
 timestamp data) GUID.
 @property {string} adType - Gets the type of the current ad.
 @property {Array.<AdVerification>} AdVerification - List of all ad verification vendors (if configured, whitelisted
 vendors only).
 @property {Array.<ClosedCaption>} closedCaptions - List of all closed captions.
 @property {Array.<CompanionAd>} Companions - List of all companion ads.
 @property {Object} CreativeIds - Key value pair from all creative ids.
 @property {number} currentTime - Gets the current time for the ad.
 @property {number} duration - Gets the duration time for the ad.
 @property {Object} error - Description object of the most recent ad error.
 @property {Array.<Extension>} Extension - List of all extensions.
 @property {boolean} hasClickThrough - True if the current ad contains a ClickThrough URL.
 @property {number} height - Gets the height of the ad.
 @property {Object} icons - Object of all industry icons.
 @property {boolean} isExpanded - Indicates whether the ad is expanded.
 @property {boolean} isInteractive - True if the current ad has been specified as interactive.
 @property {boolean} isLinear - Indicates whether the ad is linear.
 @property {boolean} isSkippable - Indicates whether the ad is skippable.
 @property {Object} linear - With details about the linear ad.
 @property {Object} macroList - Snapshot of the VAST Macros.
 @property {Array.<NonLinearAd>} nonLinear - List of all nonLinear ads.
 @property {number} remainingTime - Gets the remaining time for the ad.
 @property {number} skipOffset - Gets the skip offset time for the ad.
 @property {string} variant - Gets the variant type of the current ad.
 @property {Object} viewability - Overview about the status of visibility.
 @property {number} volume - Gets the volume value of the ad.
 @property {number} width - Gets the width of the ad.
 @property {Array.<Object>} wrappers - List of Wrapper objects, starting with the initial request up to the
 lastWrapper before the Inline tag.
 */
export declare interface AdCreative {
    captions: ClosedCaption[];
    creativeTrackers: TrackingEvents;
    icons: {
        [key: string]: IconProgram;
    } | object;
    mediaFiles: MediaFile[];
    selectedCreativeFile: Resource | null;
    selectedMediaFile: MediaFile;
}

export declare interface AdEvent {
    type: string;
    timestamp: number;
    issuer: string;
}

export declare class AdPlaylistAPI extends EventDispatcher {
    config: URLConfig;
    environmentVars: EnvironmentVars;
    error: any;
    static NAME: string;
    /**
     * @param {Object} playerFacade extended from FacadeBase
     * @param {Object} adsAPI is a reference to the session AdSlotAPI
     */
    constructor(playerFacade: PlayerFacade, adsAPI: AdSlotAPI);
    /**
     * Starts requesting a VMAP from the ad server
     *
     * @param {AdPlaylistSetup} playlistSpecs - Ad configuration for the upcoming content video.
     * @param {EnvironmentVars} environmentVars - Defines environment conditions for the upcoming ad break.
     *  https://interactiveadvertisingbureau.github.io/vast/vast4macros/vast4-macros-latest.html
     * @returns {Promise<String|Error>} Resolves if loading the VMAP has been completed
     */
    initAdPlaylist(playlistSpecs: AdPlaylistSetup, environmentVars?: EnvironmentVars): Promise<string>;
    /**
     * Starts playback and monitoring of the content and automatically inserts ad-slots
     * based on the `timeOffset` values of each VMAP AdBreak.
     * AdBreaks with a `timeOffset` equally to `0`, `"start"` or `"0%"` get requested immediately.
     *
     * @returns {Promise} Resolves if the playback started
     */
    startAdPlaylist(): Promise<void>;
    /**
     * Immediately stops monitoring content playback.
     * AdBreaks that have already started remain until they complete. If necessary, they must be canceled
     * via the `AdSlotAPI.stopAdSlot()`.
     *
     * @returns {Promise<void>} - Whether the ad slot could be closed and cleaned up or not.
     */
    stopAdPlaylist(): Promise<void>;
    dispose(): Promise<void>;
    /**
     * Requests details about the most recent error that has occurred on the AdPlaylistAPI.
     *
     * @returns {error} Description object of the most recent error or `null`.
     */
    get errorAPI(): ErrorStatus | null;
    /**
     * Returns the VMAP that has been collected for the current (content) session
     *
     * @returns {XML} XML as live parsable DOM representation, querySelector and similar work on this
     */
    get rawVMAP(): Document | null;
}

/**
 * Ad configuration for the upcoming content video.
 * @interface AdPlaylistSetup
 *
 * @property {AdSlotSetup} [preroll] - Separation Clip and prefetching setup for preroll ads.
 * @property {AdSlotSetup} [midroll] - Separation Clip and prefetching setup for midroll ads.
 * @property {AdSlotSetup} [postroll] - Separation Clip and prefetching setup for postroll ads.
 * @property {{ prefetchOffset?: number }} [overlay] - Prefetching setup for overlays.
 * @property {string} [tag] - VMAP request URL.
 * @property {string} [response] - Direct VMAP XML response.
 * @property {boolean} [strictOffsets] - If `true` offsets are maintained accordingly when fast-forwarding, no
 * overspooled ad is delivered.
 * Defaulting to `false` if omitted and ignored.
 */
export declare interface AdPlaylistSetup {
    preroll?: AdSlotSetup;
    midroll?: AdSlotSetup;
    postroll?: AdSlotSetup;
    overlay?: {
        prefetchOffset?: number;
    };
    response?: string;
    strictOffsets?: boolean;
    tag?: string;
}

export declare interface AdReinsertionHomad {
    enabled: boolean;
    setup: {
        globalConfig: string;
        clientConfig: string;
        activationCallback: Function;
        enableSessionCoverage: boolean;
        forceActivation: boolean;
    };
}

export declare interface AdReinsertionHomadUtils extends AdReinsertionHomad {
    utils: any;
}

export declare class AdSlotAPI extends EventDispatcher {
    adBreak: AdBreakSetup;
    environmentVars: EnvironmentVars;
    error: any;
    config: URLConfig;
    static NAME: string;
    static EVENT: {
        [key: string]: string;
    };
    private sequenceData;
    /**
     * @param {Object} playerFacade - A player proxy class
     * @param {Object} config - scc init configuration object
     */
    constructor(playerFacade: PlayerFacade, config: URLConfig);
    /**
     * Removes all event listeners that the SDK itself had previously added.
     * @returns {void}
     */
    dispose(): void;
    /**
     * Invokes a commercial break by requesting the specified ad setup from the ad server.
     * @param {AdBreak} adBreak - Configuration for the upcoming ad break.
     * @param {EnvironmentVars} environmentVars - Defines environment conditions for the upcoming ad break
     * @returns {Promise<String|Error>} Resolves if loading the ad manifests has been completed.
     */
    initAdSlot(adBreak: AdBreakSetup, environmentVars?: EnvironmentVars | {}): Promise<string>;
    /**
     * Starts the ad break.
     * The add-on responds by sending an `ON_AD_SLOT_START` event notifying the video player the ad break is going
     *  to start.
     */
    startAdSlot(): void;
    /**
     * Stops a running ad break.
     * The add-on responds by sending an `ON_AD_SLOT_COMPLETE` event notifying the video player it has closed and
     * cleaned up ad resources. Also the content of the video player has been reset.
     *
     * @returns {Promise}
     *  - resolve() - Whether the adSlot could be closed and cleaned up.
     *  - reject(string) - In the other case, the answer is a string.
     */
    stopAdSlot(): Promise<void>;
    /**
     * Skips (stops) a running single ad
     * The add-on responds by sending the `ON_AD_SKIPPED` event notifying the video player it has closed and cleaned up
     * a single ad.
     *
     * _Note:
     * The add-on does proceed with the next ad. Canceling the whole ad break requires calling `stopAdSlot()` instead.
     * @return {void}
     */
    skipAd(): void;
    /**
     * Following a resize of the video player, the video player calls `resize()` to allow the add-on to scale
     * and reposition itself within the display area. The add-on responds by sending an `ON_AD_SIZE_CHANGED` event.
     * @param {Rect} boundingRectangle - The video content display as a Rectangle {width, height, top, left}
     * @param {string} viewMode - Current viewing mode (normal, fullscreen, thumbnail, dynamic)
     * @return {void}
     */
    resize(boundingRectangle: Rect, viewMode: string): void;
    /**
     * Following a click on the ad, the video player calls `clickThrough()` to allow the add-on
     * to request the advertisers landing page and call the ClickThru tracking requests.
     * @param {Boolean} playerHandles - Whether the player takes responsibility for opening the landing page URL.
     * Defaulting to `false` if omitted.
     * @return {string|void} The landing page URL.
     */
    clickThrough(playerHandles?: boolean): string | void;
    /**
     * Following a click on an Industry Icon ad, the video player calls `iconClickThrough()` to allow the add-on
     * to request the advertisers landing page and call the ClickThru tracking requests.
     * @param {string} iconProgram - The program represented in the icon
     * @param {Boolean} playerHandles - Whether the player takes responsibility for opening the landing page URL.
     * Defaulting to `false` if omitted.
     * @return {string} The landing page URL.
     */
    iconClickThrough(iconProgram: string, playerHandles?: boolean): null | string | undefined;
    /**
     * The view tracking for icons is used to track when the icon creative is displayed.
     * @param {string} iconProgram The program represented in the icon
     * @returns {void}
     */
    iconView(iconProgram: string): void;
    /**
     * Requests the view trackers when a creative resource is displayed. The method should be called for each resource
     *  file of a `Companion Ad`.
     * @param {string} companionId Value of the `id` attribute from the `CompanionAd` object.
     * @returns {void}
     */
    companionView(companionId: string): void;
    /**
     * Following a click on a Companion ad, the video player calls `companionClickThrough()` to allow the add-on
     * to call the ClickThru tracking requests and request the advertisers landing page (most likely with
     * `StaticResource` creative resource only).
     * @param {string} companionId Value of the `id` attribute from the `CompanionAd` object.
     * @param {Boolean} playerHandles Whether the player takes responsibility for opening the landing page URL.
     * Defaulting to `true` since most companion creatives do handle clickthroughs by their own (`HTMLResource`,
     *  `IFrameResource`).
     * @return {string} The landing page URL.
     */
    companionClickThrough(companionId: string, playerHandles?: boolean): string | undefined;
    /**
     * Pauses the current ad break. The add-on responds by sending the `ON_AD_PAUSED` event.
     */
    pauseAd(): void;
    /**
     * Resumes the current ad break. The add-on responds by sending the `ON_AD_PLAYING` event.
     */
    resumeAd(): void;
    dispatchEvent(type: string): void;
    /**
     * Returns information about the current ad, like companions, icons, variations, verifications and more.
     * The contents of variations depend on the advertising medium, the following things can be found there:
     *
     * @returns {PublicAdAPI|null} Represents static information about the current ad.
     */
    get ad(): PublicAdAPI | null;
    /**
     * Returns information about the current ad-break, like times and more.
     *
     * @returns {PublicAdSlot|null} Represents static information about the current ad-break.
     */
    get adSlot(): PublicAdSlotAPI | null;
    /**
     * Requests details about the most recent error that has occurred on the AdSlotAPI.
     *
     * @returns {error} Description object of the most recent error or `null`.
     */
    get errorAPI(): ErrorStatus | null;
    /**
     * Returns all of the collected XML that has been collected for the whole slot
     *
     * @returns {XML} XML as live parsable DOM representation, querySelector and similar work on this.
     */
    get rawXML(): Document | null;
}

export declare interface AdSlotSetup {
    closer?: SeparationClips;
    divider?: SeparationClips;
    opener?: SeparationClips;
    prefetchOffset?: number;
}

export declare interface AdVerification {
    javascriptResource: {
        apiFramework: string | null;
        browserOptional: boolean;
        url: string | null;
    };
    tracking: TrackingEvents;
    vendor: string;
    verificationParameters: string | Element;
}

export declare interface AdVerificationSetup {
    accessMode: string;
    disabled: boolean;
    omidServiceWindow: any;
    playerHandles: boolean;
    serviceWindow: any;
    sessionCleanupDelay: number;
    timeoutScriptLoad: number;
    vendorWhitelist: string[];
}

export declare interface BaseCreative {
    adParameters: string | null;
    apiFramework: string | null;
    clickThroughURL: string | null;
    clickTracker: ClickTracker[] | CreativeViewTracker[] | Tracker[];
    resources: Resource[];
}

export declare interface ClickTracker {
    id: null | string;
    requested?: boolean;
    url: null | string;
}

export declare interface ClosedCaption {
    fileURL: string | null;
    language: string | null;
    linearity: string;
    type: string | null;
}

export declare interface CoreConfig {
    adRequestMethod: string;
    adReinsertion: {
        homad: {
            enabled: boolean;
            setup: {
                globalConfig: string;
                clientConfig: string;
                activationCallback: Function;
                enableSessionCoverage: boolean;
                forceActivation: boolean;
            };
        };
    };
    adSourceSelector: ((arg0: Resource[]) => Promise<Resource | null>) | null;
    adVerification: AdVerificationSetup;
    blockInsecureURL: boolean;
    maxWrapperDepth: number;
    playerHandlesNonLinear: boolean;
    shareVideoElementWithVPAID: boolean;
    skipOffset: number;
    timeouts: {
        adRequest: number;
        adRequestChain: number;
        adSlotRequest: number;
        contentRequest: number;
        nonlinearMaxDuration: number;
        videoRequest: number;
        vpaidReadiness: number;
        vpaidRequest: number;
    };
    trackImpressionOnStart: boolean;
    trackSkippedQuartiles: boolean;
    viewability: {
        area: number;
        time: number;
    };
}

export declare interface CreativeIds {
    adId: string | null;
    id: string | null;
}

export declare interface CreativeViewTracker {
    requested?: boolean;
    url: string;
}

export declare interface CurrentContentSource {
    currentTime: number | undefined;
    progressive: string | undefined;
}

export declare interface DefaultMimeTypes {
    VPAID: string;
    MP4: string;
    WEBM: string;
    OGG: string;
    MP3: string;
}

/**
 * @type {Object} ElementSize
 * @property {Object} boundingRect A rectangle that describes the size and position of the element
 *  (see HTML getBoundingClientRect)
 * @property {string} viewMode Either `normal` or `fullscreen`
 */
export declare interface ElementSize {
    boundingRect: Rect;
    viewMode: string;
}

export declare class EnrichmentModel {
    constructor();
    setDefaultMacros(): void;
    /**
     * Returns all macros as defined by the IAB and its encoded values.
     * @returns {Object} list of macro value pairs
     */
    getMacroList(): {
        [key: string]: any;
    };
    /**
     * Returns all macros as defined by the IAB, but surrounded by square brackets,
     *  and its encoded values.
     * @returns {Object} list of macro value pairs
     */
    getIABMacroList(): {
        [key: string]: any;
    };
    /**
     * Key and Value have to be set everytime,
     * macroToReplace needs to be set at least once
     * transformFunction is a function to be called each time the macro is replaced,
     * so you can set a value like this
     *      set('size', {width:w, height:h}, ['WxH'], function(obj){return obj.width+'x'+obj.height;}
     * while on updating the value only
     *      set('size', {width:w, height:h});
     * is needed
     *
     * @param {String} key reference key
     * @param {*} value usually a string, however transformFunction can make this a string
     * @param {(String|String[])} [macroToReplace] the macro to replace
     * @param {Function} [transformFunction] value will sent through this function on each apply call
     * @return {EnrichmentModel} to set() functions can be chained
     */
    set(key: string, value: any, macroToReplace?: string | string[], transformFunction?: Function): this;
    get(key: null | number | string | undefined, encoded?: boolean): string | PlayerDimension | null | undefined;
    setFromObject(macroObject: Record<string, any> | null): void;
    /**
     * Does replace all occurrences of macro replacement strings with values.
     * @param {string} inputString - This is most likely a URL
     * @returns {string|*} - Enriched URL
     */
    apply(inputString?: null | number | string | undefined): string;
    replaceMacroWithValue(inputString: string, macro: string, value: string): string;
    getRegExp(captureGroup: string): RegExp;
}

/**
 * @type {Object} EnvironmentVars Used for passing implementation-specific runtime variables.
 * @property {HTMLElement} adContainer The element that should be used to deliver non-video creative files.
 * Defaulting to `null` if omitted.
 * @property {number} desiredBitrate Indicates the desired bitrate in kilobits per second (kbps). The player uses this
 *  information to select Mediafiles with an appropriate bitrate. Defaulting to `null`.
 * @property {Array.<string>} desiredMimeTypes Declaration of accepted media types. Defaulting to types that the
 *  browser eventually is able to play if not set. Types must be compliant with the
 *  [IANA Media Type standard](https://www.iana.org/assignments/media-types/media-types.xhtml)
 * @property {Array.<string>} desiredVPAIDVendors Declaration of accepted VPAID vendors (whitelist). Each specified
 *  String has to represent a fully qualified domain name (FQDN), including both second-level (SLD) and top-level (TLD) domains.
 *  If omitted, the plugin tries to deliver VPAID creatives from each vendor.
 * @property {Dimension} deviceScreenSize Maximum size we can achieve when entering fullscreen.
 * @property {string} deviceType Can be either `mobile`, `tablet`, `desktop` or `tv`. Defaulting to `null`.
 * @property {string} networkReachability Can be either `wan`, `wifi` or `unreachable`. Defaulting to `null`.
 * @property {Object} vastMacros - VAST macros and their values as defined in
 *  https://interactiveadvertisingbureau.github.io/vast/vast4macros/vast4-macros-latest.html
 * @property {boolean} prefetchBuffetAds In order to replace invalid ads in a pod instantly at runtime, buffet ads from
 *  wrapper responses must be loaded up-front. `prefetchBuffetAds` determines whether stand-alone ads (buffet) will be
 *  loaded together with the pod of ads or not. Defaulting to `false` if omitted.
 */
export declare interface EnvironmentVars {
    adContainer?: HTMLElement | null;
    desiredBitrate?: number;
    desiredMimeTypes?: string[];
    desiredVPAIDVendors?: string[];
    deviceScreenSize?: string | null;
    deviceType?: string | null;
    networkReachability?: string | null;
    omidServiceWindow?: any;
    prefetchBuffetAds?: boolean;
    timeouts?: {
        adRequest?: number;
        adRequestChain?: number;
        adSlotRequest?: number;
        contentRequest?: number;
        nonlinearMaxDuration?: number;
        videoRequest?: number;
        vpaidReadiness?: number;
        vpaidRequest?: number;
    };
    vastMacros?: {
        [key: string]: any;
    };
}

export declare interface ErrorStatus {
    code?: number | undefined;
    message?: string | undefined;
    timestamp?: number | null;
}

/**
 * A class for dispatching events and managing event listeners.
 */
export declare class EventDispatcher {
    /**
     * Adds an event listener for the given type.
     * @param {string} type - The type of the event to listen for.
     * @param {Function} listener - The function to call when the event is dispatched.
     * @returns {Function} A function that can be called to remove the listener.
     */
    addEventListener(type: string, listener: EventHandler): () => void;
    /**
     * Dispatches an event of the given type with the given data.
     * @param {string} type - The type of the event to dispatch.
     */
    dispatchEvent(type: string): void;
    /**
     * Checks whether an event listener has been added for the given type and listener function.
     * @param {string} type - The type of the event to check for the listener.
     * @param {Function} listener - The listener function to check for.
     * @returns {boolean} True if the listener has been added for the given type, false otherwise.
     */
    hasEventListener(type: string, listener: EventHandler): boolean;
    /**
     * Removes an event listener for the given type.
     * @param {string} type - The type of the event to remove the listener for.
     * @param {Function} listener - The listener function to remove.
     */
    removeEventListener(type: string, listener: EventHandler): void;
}

export declare interface EventHandler {
    (event: AdEvent): void;
}

export declare interface Extension {
    extensionXML: Element | null;
    type: string | null;
}

/**
 * A player proxy class
 * @module FacadeBase
 * @class
 *
 * @property {Object} PLAYER_EVENT_MAP The implementation can be found in the documentation.
 */
export declare class FacadeBase {
    currentContentSource: CurrentContentSource;
    PLAYER_EVENT_MAP: Object;
    constructor();
    /**
     * Sets up a function that will be called whenever the specified event is delivered to the video player.
     * @param {string} type - A case-sensitive string representing the player event type to listen for.
     * @param {Function} callback - The function which receives a notification when an event of the specified type occurs.
     *
     * _*Note:*
     *
     * The PLAYER_EVENT_MAP should be defined within the Facade (constructor) to allow mapping and usage of equivalent
     * events for non-HTML5 video players. The key represents the event from the custom player, while the value
     * corresponds to the standard HTML5 event. It must also be set for HTML5 players as this is no longer part of the
     * FacadeBase!
     */
    addEventListener(type: string, callback: Function): void;
    /**
     * Requests the ad playhead position from the video player.
     * @returns {Number} Total time in seconds.
     */
    getAdCurrentTime(): number;
    /**
     * Requests the ad-duration from the video player.
     * @returns {Number} Total time in seconds. _*Note:* The time can include milliseconds (3.1234)
     */
    getAdDuration(): number;
    /**
     * Requests the current playhead position from the video player.
     * @returns {Number} Current time in seconds. _*Note:* The time can include milliseconds (3.1234)
     */
    getContentCurrentTime(): number;
    /**
     * Requests the content-duration from the video player.
     * @returns {Number} Total time in seconds. _*Note:* The time can include milliseconds (3.1234)
     */
    getContentDuration(): number;
    /**
     * @returns {CurrentContentSource} SourceObject - Current content source and playhead progress
     */
    getCurrentContentSource(): CurrentContentSource;
    /**
     * Requests the latest status of the video player, site, and other external factors.
     * @param {string} adBreakType - Linearity of the upcoming ad slot. See [`AdBreak.type`](setup-sequential-config#adbreak+type)
     * @returns {EnvironmentVars} Object describing surrounding conditions and the expected ad properties.
     */
    getEnvironmentVars(adBreakType: string): EnvironmentVars;
    /**
     * Requests the player element.
     * @returns {HTMLAudioElement|HTMLVideoElement|null} The video or audio element or null.
     *
     * _*Note:* If the player takes care of the OM-SDK, config.adVerification.playerHandles = true ._
     */
    getPlayerElement(): HTMLAudioElement | HTMLVideoElement | null;
    /**
     * Requests the current size and viewMode from the video player.
     * @returns {ElementSize} Current player dimension specs
     */
    getPlayerSize(): ElementSize;
    /**
     * Requests the current size and viewMode from the viewport that surrounds the video player.
     * @returns {ElementSize} The dimension specs of the element
     */
    getViewportSize(): ElementSize;
    /**
     * Requests the current volume level from the video player.
     * @returns {Number} The current volume level between 0-1
     */
    getVolume(): number;
    /**
     * Requests the player facade version (API version) to ensure the particular versions of `smartclientcore` and player facade
     * are supported.
     * @param {string} apiFacadeVersion - Required API version
     * @returns {string} Supported API version
     */
    handshakeVersion(apiFacadeVersion: string): string;
    /**
     * Requests the video player to mute playback.
     * @returns {Promise} Resolves if the video player has been muted
     */
    mute(): Promise<void>;
    /**
     * Requests the video player to unmute playback.
     * @returns {Promise} Resolves if the video player has been unmuted
     */
    unmute(): Promise<void>;
    /**
     * Requests the current screen state from the video player.
     * @returns {Boolean} Fullscreen state
     */
    isFullscreen(): boolean;
    /**
     * Requests the current muted state from the video player.
     * @returns {Boolean} Muted state
     */
    isMuted(): boolean;
    /**
     * Requests the current pause state from the video player.
     * @returns {Boolean} Paused state
     */
    isPaused(): boolean;
    /**
     * Requests the video player to load and play the specified media file.
     * @param {string} url - The URI of the media file to be played.
     * @param {Boolean} autoPlay - Whether the requested media file starts playback immediately.
     * @param {Boolean} disableSeeking - Whether the requested source can be seeked back and forth.
     * @returns {Promise<void|number>} That resolves if the player accepts the media file request. If an error occurs, the
     *  Promise may reject with an optional error code.
     */
    load(url: string, autoPlay?: boolean, disableSeeking?: boolean): Promise<void>;
    /**
     * Requests the video player to pause playback.
     * @returns {Promise} Resolves if the video player has been paused
     */
    pause(): Promise<void>;
    /**
     * Requests the video player to start (resume) playback.
     * @returns {Promise} Resolves if the video player has been resumed
     */
    play(): Promise<void>;
    /**
     * Removes an event listener from the video player that has been registered with `addEventListener()`
     * _*Note:* The event listener to be removed must match with the listener that has been added._
     * @param {string} type - A case-sensitive string representing the player event type to listen for.
     * @param {Function} callback - The function which receives a notification when an event of the specified type occurs.
     */
    removeEventListener(type: string, callback: Function): void;
    /**
     * restoreContentState is called right after the ad playback has finished.
     * Resumes playback by restoring the last known playback state and communicates with the AdSlotAPI upon completion.
     *
     * _*Note:* Check whether the content was really replaced by an ad. Otherwise, there should be nothing to do.
     * @returns {Promise} Resolves if the player accepts the play request; otherwise, rejects.
     */
    restoreContentState(): Promise<void>;
    /**
     * saveContentState is called right before the ad playback should begin. Notes the current playback state for
     *  later use with `restoreContentState()` and communicates with the AdSlotAPI  upon completion.
     *
     * _*Note:* Promise.resolve() without the source and the current time of the content will not work.
     * @returns {Promise} Resolves with the current content source and playhead progress for VAST tracking enrichment.
     */
    saveContentState(): Promise<CurrentContentSource>;
    /**
     * Requests the video player to change the playhead position to the specified time.
     * @param {Number} offset - Requested playhead position
     * @returns {Promise} Resolves if the video player has finished seeking to the specified position
     */
    seek(offset: number): Promise<void>;
    /**
     * Requests the video player to change the volume.
     * @param {Number} volume - Requested volume level between 0-1
     * @returns {Promise} Resolves if the video player accepted the volume change request
     */
    setVolume(volume: number): Promise<void>;
}

export declare interface FacadeBase_2 {
    currentContentSource: CurrentContentSource;
    addEventListener(type: string, callback: Function): void;
    getAdCurrentTime(): number;
    getAdDuration(): number;
    getContentCurrentTime(): number;
    getContentDuration(): number;
    getCurrentContentSource(): CurrentContentSource;
    getEnvironmentVars(adBreakType: string): EnvironmentVars;
    getPlayerElement(): HTMLAudioElement | HTMLVideoElement | null;
    getPlayerSize(): ElementSize;
    getViewportSize(): ElementSize;
    getVolume(): number;
    handshakeVersion(apiFacadeVersion: string): string;
    isFullscreen(): boolean;
    isMuted(): boolean;
    isPaused(): boolean;
    mute(): Promise<void>;
    unmute(): Promise<void>;
    load(url: string, autoPlay?: boolean, disableSeeking?: boolean): Promise<void>;
    pause(): Promise<void>;
    paused: boolean;
    play(): Promise<void>;
    removeEventListener(type: string, callback: Function): void;
    restoreContentState(): Promise<void>;
    saveContentState(): Promise<CurrentContentSource>;
    seek(offset: number): Promise<void>;
    setVolume(volume: number): Promise<void>;
    volume: number;
}

export declare interface IconProgram extends BaseCreative {
    altText: string | null;
    creativeViewTracker: CreativeViewTracker[];
    duration: number;
    hasCreative: boolean;
    height: string | null;
    hoverText: string | null;
    offset: number | null;
    program: string;
    pxratio: number;
    resources: Resource[];
    width: string | null;
    xPosition: string | null;
    yPosition: string | null;
}

export declare enum LOGLEVEL {
    SILENT = 0,
    ERROR = 1,
    NORMAL = 2,
    VERBOSE = 3
}

/**
 @type MainController
 @property {getter} AdSlotAPI - API to work with VAST responses.
 @property {getter} AdPlaylistAPI - API to work with VMAP responses.
 */
export declare class MainController {
    adPlaylistAPI: AdPlaylistAPI | null;
    adSlotAPI: AdSlotAPI | null;
    config: CoreConfig;
    static NAME: string;
    constructor(config: CoreConfig, playerFacade: PlayerFacade);
    /**
     * @returns {AdSlotAPI} - API to work with VAST responses.
     */
    get AdSlotAPI(): AdSlotAPI;
    /**
     * @returns {AdPlaylistAPI} - API to work with VMAP responses.
     */
    get AdPlaylistAPI(): AdPlaylistAPI;
}

export declare interface MediaFile {
    apiFramework: string | null;
    bitrate: number;
    deliveryType: string;
    fileURL: string | null;
    height: number;
    id: string;
    isSIMID: boolean;
    isVPAID: boolean;
    maintainAspectRatio: boolean;
    maxBitrate: number;
    mimeType: string;
    minBitrate: number;
    scalable: boolean;
    width: number;
}

export declare interface NonLinearVariation extends BaseCreative {
    creativeTrackers: TrackingEvents;
    displayDuration: null | number;
    expandedHeight: number;
    expandedWidth: number;
    height: number;
    id: string | null;
    maintainAspectRatio: boolean;
    minSuggestedDuration: number;
    selectedResource: Resource;
    scalable: boolean;
    width: number;
}

export declare interface PlayerDimension {
    height: number;
    width: number;
}

export declare interface PlayerFacade extends FacadeBase_2 {
    [key: string]: any;
}

export declare interface PublicAdAPI {
    readonly adId: string | string[] | null;
    readonly adIds: Array<string | null>;
    readonly adServingId: string | null;
    readonly closedCaptions: Array<ClosedCaption>;
    readonly companions: Array<any>;
    readonly creativeIds: CreativeIds;
    readonly currentTime: number;
    readonly duration: number | null;
    readonly error: ErrorStatus;
    readonly extensions: Array<Extension>;
    readonly hasClickThrough: boolean;
    readonly height: number;
    readonly icons: {
        [key: string]: IconProgram;
    } | object;
    readonly ids: Array<string | null>;
    readonly isExpanded: boolean;
    readonly isInteractive: boolean;
    readonly isLinear: boolean;
    readonly isSkippable: boolean;
    readonly linear: AdCreative;
    readonly macroList: object;
    readonly nonLinear: Array<NonLinearVariation> | [];
    readonly remainingTime: number;
    readonly skipOffset: number;
    readonly title: string;
    readonly type: string;
    readonly universalAdId: UniversalAdId;
    readonly universalAdIds: UniversalAdId[];
    readonly variant: string;
    readonly verifications: Array<AdVerification>;
    readonly viewability: Viewability;
    volume: number;
    readonly width: number;
    readonly wrappers: Array<Wrapper>;
}

/**
 @type PublicAdSlotAPI
 @property {number} currentClip - Current clip position.
 @property {number} currentCommercial - Current commercial position.
 @property {number} totalCommercials - Total parsed commercials from VAST.
 @property {number} totalCommercialsCurrentTime - Total playable commercials from VAST.
 @property {number} totalCommercialsDuration - Total duration of all playable commercials.
 @property {number} totalClips - Total clip count.
 @property {number} totalPlayableClips - Total playable clip count.
 */
export declare interface PublicAdSlotAPI {
    readonly currentClip: number;
    readonly currentCommercial: number;
    readonly totalCommercials: number;
    readonly totalCommercialsCurrentTime: number;
    readonly totalCommercialsDuration: number;
    readonly totalClips: number;
    readonly totalPlayableClips: number;
    readonly totalPlayableCommercials: number;
}

/**
 * Represents a rectangle with position and size properties.
 * This interface is used to describe the size and position of an element,
 * similar to the DOMRect interface in the Web API.
 *
 * @interface Rect
 * @see {@link https://developer.mozilla.org/en-US/docs/Web/API/DOMRect}
 *
 * @property {number} [bottom] - The y-coordinate of the bottom of the rectangle.
 *  It has the same value as `y + height`, or `y` if `height` is negative.
 * @property {number} height - The height of the rectangle. This is a required property.
 * @property {number} [left] - The x-coordinate of the left side of the rectangle.
 *  It has the same value as `x`, or `x + width` if `width` is negative.
 * @property {number} [right] - The x-coordinate of the right side of the rectangle.
 *  It has the same value as `x + width`, or `x` if `width` is negative.
 * @property {number} [top] - The y-coordinate of the top of the rectangle.
 *  It has the same value as `y`, or `y + height` if `height` is negative.
 * @property {number} width - The width of the rectangle. This is a required property.
 * @property {number} x - The x-coordinate of the origin of the rectangle. This is a required property.
 * @property {number} y - The y-coordinate of the origin of the rectangle. This is a required property.
 */
export declare interface Rect {
    bottom?: number;
    height: number;
    left?: number;
    right?: number;
    top?: number;
    width: number;
    x: number;
    y: number;
}

export declare interface Resource {
    creativeType: string | null;
    resource: string | null;
    type: string;
}

export declare interface SeparationClips {
    dash?: string;
    forcePlay?: boolean;
    hls?: string;
    progressive: string;
}

export declare class SmartClientCore {
    private error;
    static NAME: string;
    /**
     * Sets up a new core instance with the given configuration, as specified in Global Configuration.
     * Returns a promise that either resolves with the APIs or rejects with a facade error.
     *
     * @param {PlayerFacade} playerFacade - The player facade object to establish a connection with.
     * @param {config} config - Basic configuration for the APIs (core instance)
     * @returns {Promise<MainController|Error>} Resolves with MainController - Reject with Error (Facade failure)
     */
    init(playerFacade: PlayerFacade, config: CoreConfig): Promise<MainController>;
    /**
     * @returns {number} Ad request methods
     */
    get AD_REQUEST_METHOD(): typeof AD_REQUEST_METHOD;
    /**
     * @returns {Object} Ad types
     */
    get AD_TYPE(): typeof AD_TYPE;
    /**
     * @returns {Object} Ad variants
     */
    get AD_VARIANT(): typeof AD_VARIANT;
    /**
     * @returns {Object} the default config.
     */
    get CONFIG(): CoreConfig;
    /**
     * @returns {Object} the default mimetypes.
     */
    get DEFAULT_MIMETYPES(): DefaultMimeTypes;
    /**
     * @returns {Object} List of event types as ENUMs
     */
    get EVENT(): object;
    /**
     * @returns {Array} List of event types
     */
    get EVENTS(): Array<string>;
    get LOGLEVEL(): typeof LOGLEVEL;
    /**
     * Set the log level.
     * @param {LOGLEVEL} level - The log level to set. ( SILENT - 0, ERROR - 1, NORMAL - 2, VERBOSE - 3)
     * @returns {void}
     */
    set debugLevel(level: LOGLEVEL.SILENT | LOGLEVEL.ERROR | LOGLEVEL.NORMAL | LOGLEVEL.VERBOSE);
    /**
     * Get the current log level.
     * @returns {number} Returns '0' by default, or the value set by the 'debugLevel' setter.
     */
    get debugLevel(): LOGLEVEL.SILENT | LOGLEVEL.ERROR | LOGLEVEL.NORMAL | LOGLEVEL.VERBOSE;
    /**
     * Get the debug log object.
     * @returns {Object} Debug log object with log functions.
     */
    get debugger(): object;
    /**
     * @returns {string} Facade version string
     */
    get facadeVersion(): string;
    /**
     * @returns {Object} InitError object
     */
    get initError(): Object;
    /**
     * @returns {Array} Array of supported Facades by version
     */
    get supportedFacadeVersions(): Array<string>;
    /**
     * @returns {string} __libname__ version string
     */
    get version(): string;
}

export declare const smartclientcore: SmartClientCore;
export default smartclientcore;

export declare interface Tracker {
    offset?: number | string;
    requested?: boolean;
    url: string;
}

export declare interface TrackingEvents {
    [key: string]: Tracker[];
}

export declare interface UniversalAdId {
    idRegistry: string | null;
    idValue: string | null;
}

export declare interface URLConfig extends CoreConfig {
    adReinsertion: {
        homad: AdReinsertionHomadUtils;
    };
    bitrate: number;
    enrichment: EnrichmentModel;
    issuer: string;
    isWrapper: boolean;
    response: any;
    urlhandler: any;
    viewMode: string;
}

/**
 * Contains information about the visibility status, the corresponding threshold value and the
 * current visibility value.
 * @returns {Object} Overview about the status of visibility.
 */
export declare interface Viewability {
    state: boolean;
    threshold: number;
    value: number;
}

/**
 * Represents a wrapper containing information about a loading process.
 */
export declare interface Wrapper {
    loadEnd: string;
    loadStart: string;
    loadTime: number;
    url: string | null;
}

export { }
