/**
 * @typedef {Object} MediaError https://developer.mozilla.org/en-US/docs/Web/API/MediaError
 * @property {number} code A number which represents the general type of error that occurred.
 * @property {string} message A human-readable string which provides specific diagnostic information
 * to help the reader understand the error condition which occurred.
 */

/**
 * @typedef {Object} ElementSize
 * @property {Object} boundingRect A rectangle that describes the size and position of the element (see HTML getBoundingClientRect)
 * @property {string} viewMode Either `normal` or `fullscreen`
 */

/**
 * @typedef {Object} EnvironmentVars Used for passing implementation-specific runtime variables.
 * @property {number} desiredBitrate Indicates the desired bitrate in kilobits per second (kbps). The player uses this
 *  information to select Mediafiles with an appropriate bitrate. Defaulting to `null`.
 * @property {Array.<string>} desiredMimeTypes Declaration of accepted media types. Defaulting to types that the
 *  browser eventually is able to play. Types must be compliant with the
 *  [IANA Media Type standard](https://www.iana.org/assignments/media-types/media-types.xhtml)
 * @property {Array.<string>} desiredVPAIDVendors Declaration of accepted VPAID vendors (whitelist). Each specified
 *  String has to represent an unique identifier that can be found within the creative media file URL, most likely the
 *  domain name. If omitted, the plugin tries to deliver VPAID creatives from each vendor.
 * @property {string} deviceType Can be either `mobile`, `tablet`, `desktop` or `tv`. Defaulting to `null`.
 * @property {Dimension} deviceScreenSize Maximum size we can achieve when entering fullscreen.
 * @property {string} networkReachability Can be either `wan`, `wifi` or `unreachable`. Defaulting to `null`.
 * @property {Object} vastMacros - VAST macros and their values as defined in
 *  https://interactiveadvertisingbureau.github.io/vast/vast4macros/vast4-macros-latest.html
 * @property {boolean} prefetchBuffetAds In order to replace invalid ads in a pod instantly at runtime, buffet ads from
 *  wrapper responses must be loaded up-front. `prefetchBuffetAds` determines whether stand-alone ads (buffet) will be
 *  loaded together with the pod of ads or not. Defaulting to `false` if omitted.
 * @property {HTMLElement} adContainer The element that should be used to deliver non-video creative files.
 * Defaulting to `null` if omitted.
 */

const PLAYER_FACADE_VERSION = '3.2';

/**
 * A player proxy class
 * (last change 2026-01-26T09:47:45.967Z)
 * @module FacadeBase
 * @class
 */
export class FacadeBase {

	constructor(playerReference = {}) {
		this.player = playerReference;
		this.currentContentSource = {};

		/**
		 * Map player specific events to the corresponding HTML5 MediaEvent
		 * @return {{ended: string, error: string, loadeddata: string, pause: string, play: string, timeupdate: string}}
		 */
		this.PLAYER_EVENT_MAP = {
			'ended': 'ended',
			'error': 'error',
			'playerPause': 'pause',
			'playerPlay': 'playing',
			'playerResize': 'resize',
			'loadeddata': 'loadeddata',
			'timeupdate': 'timeupdate',
			'volumeChanged': 'volumechange'
		};
	}
	
	/**
	 * Requests the player facade version (API version) to ensure the particular versions of `smartclientcore` and player facade
	 * are supported.
	 * @param {string} apiFacadeVersion - Required API version
	 * @returns {string} Supported API version
	 */
	handshakeVersion(apiFacadeVersion) {
		return PLAYER_FACADE_VERSION;
	}

	/**
	 * Requests the video player to load and play the specified media file.
	 * @param {string} url - Media file URI
	 * @param {Boolean} autoPlay - Whether the requested media file starts playback immediately.
	 * @param {Boolean} disableSeeking - Whether the requested source can be seeked back and forth.
	 * @returns {Promise} Resolves if the player accepted the media file request
	 */
	load(url, autoPlay = true, disableSeeking = false) {}

	/**
	 * Sets up a function that will be called whenever the specified event is delivered to the video player.
	 * @param {string} type - A case-sensitive string representing the player event type to listen for.
	 * @param {Function} callback - The function which receives a notification when an event of the specified type occurs.
	 */
	addEventListener(type, callback) {}

	/**
	 * Removes an event listener from the video player that has been registered with `addEventListener()`
	 * _*Note:* The event listener to be removed must match with the listener that has been added._
	 * @param {string} type - A case-sensitive string representing the player event type to listen for.
	 * @param {Function} callback - The function which receives a notification when an event of the specified type occurs.
	 */
	removeEventListener(type, callback) {}
	
	/**
	 * Should freeze the current playback state (for later use with `releaseContent()`)
	 * and respond to the AdSlotAPI once finished.
	 * @returns {Promise} Current content source and playhead progress for VAST tracking enrichment
	 */
	lockContent() {}
	
	/**
	 * Should restore the last known playback state, resume playback
	 * and respond to the AdSlotAPI once finished.
	 * @returns {Promise} Either resolves if the player accepted the play request or not
	 */
	releaseContent() {}

	/**
	 * @returns {SourceObject} Current content source and playhead progress
	 */
	getCurrentContentSource() {}

	/**
	 * Requests the current playhead position from the video player.
	 * @returns {Number} Current time in seconds.
	 */
	getCurrentTime() {}

	/**
	 * Requests the content-duration from the video player.
	 * @returns {Number} Total time in seconds.
	 */
	getDuration() {}
	
	/**
	 * Requests the latest status of the video player, site, and other external factors.
	 * @param {string} adBreakType - Linearity of the upcoming ad slot. See [`AdBreak.type`](setup-sequential-config#adbreak+type)
	 * @returns {EnvironmentVars} Description object of the most recent error or `null`.
	 */
	getEnvironmentVars(adBreakType) {}
	
	/**
	 * Requests details about the most recent error that has occurred on the video player.
	 * @returns {MediaError} Description object of the most recent error or `null`.
	 */
	getError() {}
	
	/**
	 * Requests the current size and viewMode from the video player.
	 * @returns {ElementSize} Current player dimension specs
	 */
	getPlayerSize() {}
	
	/**
	 * Requests the current size and viewMode from the viewport that surrounds the video player.
	 * @returns {ElementSize} The dimension specs of the element
	 */
	getViewportSize() {}
	
	/**
	 * Requests the video player element
	 * @returns {Element} The video element
	 * @deprecated
	 */
	getVideoElement() {}

	/**
	 * Requests the player element
	 * @returns {Element} The video or audio element
	 */
	getPlayerElement() {}

	/**
	 * Requests the video player to change the volume.
	 * @param {Number} volume - Requested volume level between 0-1
	 * @returns {Promise} Resolves if the video player accepted the volume change request
	 */
	setVolume(volume) {}

	/**
	 * Requests the current volume level from the video player.
	 * @returns {Number} The current volume level between 0-1
	 */
	getVolume() {}

	/**
	 * Requests the current muted state from the video player.
	 * @returns {Boolean} Muted state
	 */
	isMuted() {}

	/**
	 * Requests the video player to mute playback.
	 * @returns {Promise} Resolves if the video player has been muted
	 */
	mute() {}

	/**
	 * Requests the video player to unmute playback.
	 * @returns {Promise} Resolves if the video player has been unmuted
	 */
	unmute() {}

	/**
	 * Requests the current play state from the video player.
	 * @returns {Boolean} Playback state
	 */
	isPaused() {}

	/**
	 * Requests the video player to pause playback.
	 * @returns {Promise} Resolves if the video player has been paused
	 */
	pause() {}

	/**
	 * Requests the video player to start (resume) playback.
	 * @returns {Promise} Resolves if the video player has been resumed
	 */
	play() {}

	/**
	 * Requests the video player to change the playhead position to the specified time.
	 * @param {Number} offset - Requested playhead position
	 * @returns {Promise} Resolves if the video player has finished seeking to the specified position
	 */
	seek(offset) {}
}
