/**
 * Sample code: How to implement companion ads from SXP/IP
 *
 * 1. Add event handler for `ON_COMPANION_DETECTED` and `ON_EXTENSION_DETECTED`
 *	  to collect the companion ad(s) and additional data from the extension
 *    ```
 *		if (event.type === smartclientcore.EVENT.ON_COMPANIONS_DETECTED) {
 *			this.companions = this.adSlotAPI.ad.companions;
 *		}
 *
 *		if (event.type === smartclientcore.EVENT.ON_EXTENSIONS_DETECTED) {
 *			this.extensions = this.adSlotAPI.ad.extensions;
 *		}
 *    ```
 * 2. Add another event handler for ON_AD_STARTED.
 * 3. Check whether you have both data objects available and start the delivery.
 *    ```
 * 		if (event.type === smartclientcore.EVENT.ON_AD_STARTED) {
 *			if(this.companions && this.extensions) {
 *				let companionAds = new CompanionAds(this.adSlotAPI);
 *				companionAds.set(this.companions, this.extensions);
 *				companionAds.deliver();
 *			}
 *		}
 *	  ```
 */
export class CompanionAds {
	
	constructor(api) {
		this.adSlotAPI = api;
		this.companions = null;
		this.extensions = null;
	}
	
	/**
	 * @param {Array.<CompanionAd>} companions The value of this property is an Array that provides ad companion
	 *  details for each `<CompanionAd>` element.
	 * @param {Array.<Extension>} extensions The value of this property is an Array that provides ad extension
	 *  details for the `<Extensions>` element.
	 */
	set(companions, extensions) {
		this.companions = companions;
		this.extensions = extensions;
	}
	
	/**
	 * Starts delivery of one or multiple Companion Ads.
	 * @returns {void}
	 */
	deliver() {
		if (this.extensions && this.extensions.length > 0
			&& this.companions && this.companions.length > 0) {
			this.extensions.filter(this._extensionFilter.bind(this));
		}
	}
	
	/**
	 * Checks whether the VAST Extension belongs to smartclip and starts delivery.
	 * @param {Extension} extension Object consisting of `type` and `extensionXML`
	 * @returns {boolean}
	 * @private
	 */
	_extensionFilter(extension) {
		
		const $extension = extension.extensionXML;
		
		if ($extension.getAttribute('name') !== 'SXP') {
			return false;
		}
		
		let injectionScript = null;
		
		const $adInfo = $extension.querySelector('AdInfo');
		
		if ($adInfo) {
			
			let customaid = $adInfo.getAttribute('customaid');
			injectionScript = customaid.replace(/(^.*\[|\].*$)/g, '');
		}
		
		// Step 1: Append all Companion Ads
		const companionAdInfo = $extension.querySelectorAll('Companion');
		for (var i = 0; i < companionAdInfo.length; i++) {
			
			let c = companionAdInfo[i];
			
			let banner = {
				zone: c.getAttribute('zone'),
				id: c.getAttribute('id')
			};
			
			let variation = this.companions.filter((companion) => {
				
				return companion.id === banner.id;
			}).shift();
			
			variation.zone = banner.zone;
			variation.customaid = injectionScript;
			
			this._deliverCompanion(variation);
		}
		
		// Step 2: Append the injection script, only once
		if (injectionScript) {
			
			var script = document.createElement('script');
			script.src = injectionScript;
			document.body.appendChild(script);
			
			return true;
		}
		return false;
	}
	
	/**
	 *
	 * @param {CompanionAd} creative Object `zone`, `customaid`, `type`, `companionXML`
	 * @private
	 */
	_deliverCompanion(creative) {
		
		const companion = creative.companionXML;
		const companionSpecs = {
			id: creative.id,
			width: parseInt(companion.getAttribute('width'), 10),
			height: parseInt(companion.getAttribute('height'), 10),
			assetWidth: parseInt(companion.getAttribute('assetWidth'), 10),
			assetHeight: parseInt(companion.getAttribute('assetHeight'), 10),
			expandedWidth: parseInt(companion.getAttribute('expandedWidth'), 10),
			expandedHeight: parseInt(companion.getAttribute('expandedHeight'), 10),
			pxratio: parseInt(companion.getAttribute('pxratio'), 10),
			apiFramework: companion.getAttribute('apiFramework'),
			adSlotId: companion.getAttribute('adSlotId'),
			renderingMode: companion.getAttribute('renderingMode')
		};
		
		const $zone = document.querySelector('#' + creative.zone);
		
		// TODO: Decide what type of companion resource should be supported. We'll just illustrate sample code for one
		//  `IFrameResource`.
		const iframeResource = creative.resources.filter((resource) => {
			
			return resource.type === 'IFrameResource';	// other possible types are `StaticResource`, `HTMLResource`
		}).shift();
		
		if (iframeResource) {
			
			let $iframe = document.createElement('iframe');
			$iframe.src = iframeResource.resource;
			$iframe.width = companionSpecs.width;
			$iframe.height = companionSpecs.height;
			$iframe.frameBorder = 0;
			$iframe.setAttribute('scrolling', 'no');
			$iframe.setAttribute('margin', '0');
			$iframe.setAttribute('allowtransparency', 'true');
			
			// TODO: Don't forget to let us know once the companion ad becomes displayed.
			$iframe.onload = () => {
				this.adSlotAPI.companionView(creative.id);
			};
			
			// TODO: Most companion ad can provide a clickthrough of their own, like `HTMLResource` and `IFrameResource`.
			//  But in case where the creative cannot provide a clickthrough, such as with a simple static image (`StaticResource`),
			//  you have to add a click handler that requests `this.adSlotAPI.companionClickThrough(creative.id, playerHandles)`.
			//  Check the documentation for `playerHandles`.
			
			if ($zone) {
				$zone.appendChild($iframe);
			}
			else {
				console.warn(
					`[CompanionAds] Target element "#${creative.zone}" not found to deliver CompanionAd ID ${creative.id}`
				);
			}
		}
	}
}
